#!/usr/bin/perl
#
# This is an example of perl module support for the net-snmp agent.
#
# To load this into a running agent with embedded perl support turned
# on, simply put the following line (without the leading # mark) your
# snmpd.conf file:
#
#   perl do "/path/to/perl_module.pl";
#
#version: 1.0  ----- 282

my $omoid = qq|
OS TCP status:
$tcpst{'ESTABLISHED'} = .1.3.6.1.4.1.99988.10.1.2.1
$tcpst{'SYN_RECV'} = .1.3.6.1.4.1.99988.10.1.2.2.2
$tcpst{'FIN_WAIT'} = .1.3.6.1.4.1.99988.10.1.2.3
$tcpst{'TIME_WAIT'} = .1.3.6.1.4.1.99988.10.1.2.4
$tcpst{'CLOSE'} = .1.3.6.1.4.1.99988.10.1.2.5
$tcpst{'TOTAL'} = .1.3.6.1.4.1.99988.10.1.2.6

nginx runing status:
nginx_reqs_count = .1.3.6.1.4.1.99988.10.5.2.1
nginx_active_connection = .1.3.6.1.4.1.99988.10.5.2.2
nginx_connection_count =.1.3.6.1.4.1.99988.10.5.2.3

cpu used info:
cpu_user = .1.3.6.1.4.1.99988.10.6.2.1
cpu_system = .1.3.6.1.4.1.99988.10.6.2.2
cpu_idle = .1.3.6.1.4.1.99988.10.6.2.3
cpu_wait = .1.3.6.1.4.1.99988.10.6.2.4

mysql status info:
$mysqlst{'select_count'} = .1.3.6.1.4.1.99988.10.10.2.1
$mysqlst{'update_count'} = .1.3.6.1.4.1.99988.10.10.2.2
$mysqlst{'insert_count'} = .1.3.6.1.4.1.99988.10.10.2.3
$mysqlst{'delete_count'} = .1.3.6.1.4.1.99988.10.10.2.4
$mysqlst{'replace_count'} = .1.3.6.1.4.1.99988.10.10.2.5
$mysqlst{'slowsql_count'} = .1.3.6.1.4.1.99988.10.10.2.6
$mysqlst{'fullscan_count'} = .1.3.6.1.4.1.99988.10.10.2.7
$mysqlst{'slave_io'} = .1.3.6.1.4.1.99988.10.10.2.8
$mysqlst{'slave_sql'} = .1.3.6.1.4.1.99988.10.10.2.9

other state:
hdcstat = .1.3.6.1.4.1.99988.10.20.2.1

|;
my $tcpstat_at = '.1.3.6.1.4.1.99988.10.1';
my $nginx_reqs_at = '.1.3.6.1.4.1.99988.10.5';
my $cpu_used_at = '.1.3.6.1.4.1.99988.10.6';
my $mysql_at = '.1.3.6.1.4.1.99988.10.10';
my $other_at = '.1.3.6.1.4.1.99988.10.20';

BEGIN {
    print STDERR "starting perl_module.pl\n";
}

use NetSNMP::OID (':all');
use NetSNMP::agent (':all');
use NetSNMP::ASN (':all');
use LWP::UserAgent;
use Time::Local;
use DBI;


print STDERR "perl_module.pl loaded ok\n";

# set to 1 to get extra debugging information
$debugging = 0;

# if we're not embedded, this will get auto-set below to 1
$subagent = 0;

my %g_conf;
$g_conf{'nginx-status'} = 'http://127.0.0.1/server-status';
#$g_conf{'sysinfo_dir'} = '/data/logs/sys_info/';
$g_conf{'record_dir'} = '/data/logs/sys_mon/';
$g_conf{'mysql_user'} = 'root';
$g_conf{'mysql_passwd'} = '';
$g_conf{'mysql_socket'} = '/var/lib/mysql/mysql.sock';
my $conf = shift || "/etc/snmp/mm.conf";
if(open(DDD,$conf)) {
    while(<DDD>) {
        chomp;
        my ($k,$v) = (/^\s*([^#]{1}\S+)\s*=\s*(\S+)/o);
        $g_conf{$k} = $v;
    }
    close(DDD);
}


# where we are going to hook onto
my $tcpstat_oid = new NetSNMP::OID($tcpstat_at);
my $nginx_reqs_oid = new NetSNMP::OID($nginx_reqs_at);
my $cpu_used_oid = new NetSNMP::OID($cpu_used_at);
my $mysql_oid = new NetSNMP::OID($mysql_at);
my $other_oid = new NetSNMP::OID($other_at);

# If we're not running embedded within the agent, then try to start
# our own subagent instead.
if (!$agent) {
    $agent = new NetSNMP::agent('Name' => 'dalesnmp', # reads test.conf
                                'AgentX' => 1);   # make us a subagent
    $subagent = 1;
    print STDERR "started us as a subagent ($agent)\n"
}

my (%tcpst,%mysqlst);
$tcpst{'time'} = 0;
# we register ourselves with the master agent we're embedded in.  The
# global $agent variable is how we do this:
$agent->register('tcpstat',$tcpstat_oid, \&tcp_snmp_handler);
$agent->register('nginx_reqs',$nginx_reqs_oid, \&nginx_reqs_handler);
$agent->register('cpu_used',$cpu_used_oid, \&cpu_used_handler);
$agent->register('mysql',$mysql_oid, \&mysql_handler);
$agent->register('other',$other_oid, \&other_handler);


if ($subagent) {
    # We need to perform a loop here waiting for snmp requests.  We
    # aren't doing anything else here, but we could.
    $SIG{'INT'} = \&shut_it_down;
    $SIG{'QUIT'} = \&shut_it_down;
    $running = 1;
    while($running) {
        $agent->agent_check_and_process(1);  # 1 = block
        print STDERR "mainloop excercised\n" if ($debugging);
    }
    $agent->shutdown();
}

######################################################################
# define a subroutine to actually handle the incoming requests to our
# part of the OID tree.  This subroutine will get called for all
# requests within the OID space under the registration oid made above.
sub tcp_snmp_handler {
    my ($handler, $registration_info, $request_info, $requests) = @_;
    my $request;

    print STDERR "refs: ",join(", ", ref($handler), ref($registration_info), 
                               ref($request_info), ref($requests)),"\n" if ($debugging);

    print STDERR "processing a request of type " . $request_info->getMode() . "\n"
        if ($debugging);
    my $r = get_history('netstat');
    $tcpst{'ESTABLISHED'} = 0;
    $tcpst{'SYN_RECV'} = 0;
    $tcpst{'FIN_WAIT'} = 0;
    $tcpst{'TIME_WAIT'} = 0;
    $tcpst{'CLOSE'} = 0;
    $tcpst{'TOTAL'} = 0;
    foreach my $k (split(',',$r)) {
        my @t = split(':',$k);
        $tcpst{$t[0]} = $t[1];
    }
    for($request = $requests; $request; $request = $request->next()) {
      my $oid = $request->getOID();

      if ($request_info->getMode() == MODE_GET) {
        # if the requested oid is equals to ours, then return the data
        if ($oid == new NetSNMP::OID($tcpstat_at . ".2.1")) {
          $request->setValue(ASN_COUNTER64, $tcpst{'ESTABLISHED'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.2")) {
            $request->setValue(ASN_COUNTER64, $tcpst{'SYN_RECV'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.3")) {
            $request->setValue(ASN_COUNTER64, $tcpst{'FIN_WAIT'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.4")) {
            $request->setValue(ASN_COUNTER64, $tcpst{'TIME_WAIT'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.5")) {
            $request->setValue(ASN_COUNTER64, $tcpst{'CLOSE'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.6")) {
            $request->setValue(ASN_COUNTER64, $tcpst{'TOTAL'});
        }
      } elsif ($request_info->getMode() == MODE_GETNEXT) {
        # if the requested oid is lower than ours, then return ours
        if ($oid < new NetSNMP::OID($tcpstat_at . ".2.1")) {
          $request->setOID($tcpstat_at . ".2.1");
          $request->setValue(ASN_COUNTER64, $tcpst{'ESTABLISHED'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.1")) {
            $request->setOID($tcpstat_at . ".2.2");
          $request->setValue(ASN_COUNTER64, $tcpst{'SYN_RECV'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.2")) {
            $request->setOID($tcpstat_at . ".2.3");
            $request->setValue(ASN_COUNTER64, $tcpst{'FIN_WAIT'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.3")) {
            $request->setOID($tcpstat_at . ".2.4");
            $request->setValue(ASN_COUNTER64, $tcpst{'TIME_WAIT'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.4")) {
            $request->setOID($tcpstat_at . ".2.5");
            $request->setValue(ASN_COUNTER64, $tcpst{'CLOSE'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.5")) {
            $request->setOID($tcpstat_at . ".2.6");
            $request->setValue(ASN_COUNTER64, $tcpst{'TOTAL'});
        } elsif ($oid == new NetSNMP::OID($tcpstat_at . ".2.6")) {
            $request->setValue(ASN_NULL, "");
        } 
      }
    }

    print STDERR "  finished processing\n"
        if ($debugging);
}

sub nginx_reqs_handler {
    my ($handler, $registration_info, $request_info, $requests) = @_;
    my $request;

    print STDERR "refs: ",join(", ", ref($handler), ref($registration_info), 
                               ref($request_info), ref($requests)),"\n" if ($debugging);

    print STDERR "processing a request of type " . $request_info->getMode() . "\n"
        if ($debugging);
    my $ua = LWP::UserAgent->new;
    my $response = $ua->get($g_conf{'nginx-status'});
    my ($aconn,$hconn,$reqs) = 0;
    if ($response->code() == "200") {
        ($aconn,$hconn,$reqs) = ($response->content() =~ /Active connections: (\d+).+server accepts handled requests.+\d+\s+(\d+)\s+(\d+)/gso);
    }
    for($request = $requests; $request; $request = $request->next()) {
      my $oid = $request->getOID();

      if ($request_info->getMode() == MODE_GET) {
        # if the requested oid is equals to ours, then return the data
        if ($oid == new NetSNMP::OID($nginx_reqs_at . ".2.1")) {
          $request->setValue(ASN_COUNTER64, $reqs);
        } elsif ($oid == new NetSNMP::OID($nginx_reqs_at . ".2.2")) {
            $request->setValue(ASN_COUNTER64, $aconn);
        } elsif ($oid == new NetSNMP::OID($nginx_reqs_at . ".2.3")) {
            $request->setValue(ASN_COUNTER64, $hconn);
        } 
      } elsif ($request_info->getMode() == MODE_GETNEXT) {
        # if the requested oid is lower than ours, then return ours
        if ($oid < new NetSNMP::OID($nginx_reqs_at . ".2.1")) {
          $request->setOID($nginx_reqs_at . ".2.1");
          $request->setValue(ASN_COUNTER64, $reqs);
        } elsif ($oid == new NetSNMP::OID($nginx_reqs_at . ".2.1")) {
            $request->setOID($nginx_reqs_at . ".2.2");
          $request->setValue(ASN_COUNTER64, $aconn);
        } elsif ($oid == new NetSNMP::OID($nginx_reqs_at . ".2.2")) {
            $request->setOID($nginx_reqs_at . ".2.3");
            $request->setValue(ASN_COUNTER64, $hconn);
        } elsif ($oid == new NetSNMP::OID($nginx_reqs_at . ".2.3")) {
            $request->setValue(ASN_NULL, "");
        } 
      }
    }
}

sub cpu_used_handler {
    my ($handler, $registration_info, $request_info, $requests) = @_;
    my $request;

    print STDERR "refs: ",join(", ", ref($handler), ref($registration_info), 
                               ref($request_info), ref($requests)),"\n" if ($debugging);

    print STDERR "processing a request of type " . $request_info->getMode() . "\n"
        if ($debugging);
    my %pcpu;
    my $r = get_history('pcpu');
    foreach my $k (split(',',$r)) {
        my @t = split(':',$k);
        $pcpu{$t[0]} = int($t[1]);
    }

    for($request = $requests; $request; $request = $request->next()) {
      my $oid = $request->getOID();

      if ($request_info->getMode() == MODE_GET) {
        # if the requested oid is equals to ours, then return the data
        if ($oid == new NetSNMP::OID($cpu_used_at . ".2.1")) {
          $request->setValue(ASN_COUNTER64, $pcpu{'user'});
        } elsif ($oid == new NetSNMP::OID($cpu_used_at . ".2.2")) {
            $request->setValue(ASN_COUNTER64, $pcpu{'system'});
        } elsif ($oid == new NetSNMP::OID($cpu_used_at . ".2.3")) {
            $request->setValue(ASN_COUNTER64, $pcpu{'idle'});
        } elsif ($oid == new NetSNMP::OID($cpu_used_at . ".2.4")) {
            $request->setValue(ASN_COUNTER64, $pcpu{'wait'});
        } 
      } elsif ($request_info->getMode() == MODE_GETNEXT) {
        # if the requested oid is lower than ours, then return ours
        if ($oid < new NetSNMP::OID($cpu_used_at . ".2.1")) {
          $request->setOID($cpu_used_at . ".2.1");
          $request->setValue(ASN_COUNTER64, $pcpu{'user'});
        } elsif ($oid == new NetSNMP::OID($cpu_used_at . ".2.1")) {
            $request->setOID($cpu_used_at . ".2.2");  
          $request->setValue(ASN_COUNTER64, $pcpu{'system'});
        } elsif ($oid == new NetSNMP::OID($cpu_used_at . ".2.2")) {
            $request->setOID($cpu_used_at . ".2.3");
            $request->setValue(ASN_COUNTER64, $pcpu{'idle'});
        } elsif ($oid == new NetSNMP::OID($cpu_used_at . ".2.3")) {
            $request->setOID($cpu_used_at . ".2.4");
            $request->setValue(ASN_COUNTER64, $pcpu{'wait'});
        } elsif ($oid == new NetSNMP::OID($cpu_used_at . ".2.4")) {
            $request->setValue(ASN_NULL, "");
        } 
      }
    }
}

sub mysql_handler {
    my ($handler, $registration_info, $request_info, $requests) = @_;
    my $request;

    print STDERR "refs: ",join(", ", ref($handler), ref($registration_info), 
                               ref($request_info), ref($requests)),"\n" if ($debugging);

    print STDERR "processing a request of type " . $request_info->getMode() . "\n"
        if ($debugging);
    if($mysqlst{'time'} == 0 or time() - $mysqlst{'time'} > 90) {
        my $err = 0;
        for my $key (keys %mysqlst) {
            $mysqlst{$key} = 0
        }
        my $dbh;
        if (-r $g_conf{'mysql_socket'}) {
            $dbh = DBI->connect("dbi:mysql:mysql;host=localhost;port=3306;mysql_socket=". $g_conf{'mysql_socket'} .
                ";mysql_connect_timeout=10", $g_conf{'mysql_user'},$g_conf{'mysql_passwd'})
                or $err = 1;
        } else {
            $err = 1;
        }
        if( $err == 0) {
            my $sth = $dbh->prepare('show global status');
            $sth->execute;
            while(my $row = $sth->fetchrow_hashref()) {
                if( $row->{'Variable_name'} eq "Com_select" ) {
                    $mysqlst{'select_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Qcache_hits" ) {
                    $mysqlst{'select_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Com_update" ) {
                    $mysqlst{'update_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Com_insert" ) {
                    $mysqlst{'insert_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Com_insert_select" ) {
                    $mysqlst{'insert_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Com_delete" ) {
                    $mysqlst{'delete_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Com_replace" ) {
                    $mysqlst{'replace_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Com_replace_select" ) {
                    $mysqlst{'replace_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Slow_queries" ) {
                    $mysqlst{'slowsql_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Select_full_join" ) {
                    $mysqlst{'fullscan_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Select_full_range_join" ) {
                    $mysqlst{'fullscan_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Select_range" ) {
                    $mysqlst{'fullscan_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Select_scan" ) {
                    $mysqlst{'fullscan_count'} += $row->{'Value'};
                } elsif( $row->{'Variable_name'} eq "Sort_scan" ) {
                    $mysqlst{'fullscan_count'} += $row->{'Value'};
                } 
            }
            $sth->finish();
            my $sth = $dbh->prepare('show slave status');
            $sth->execute;
            while(my $row = $sth->fetchrow_hashref()) {
                if( $row->{'Slave_IO_Running'} eq "Yes" ) {
                    $mysqlst{'slave_io'} = 0;
                } elsif( $row->{'Slave_IO_Running'} eq "No" ) {
                    $mysqlst{'slave_io'} = 1;
                }
                if( $row->{'Slave_SQL_Running'} eq "Yes" ) {
                    $mysqlst{'slave_sql'} = 0;
                } elsif( $row->{'Slave_SQL_Running'} eq "No" ) {
                    $mysqlst{'slave_sql'} = 1;
                }
            }
            $sth->finish();
            $dbh->disconnect();
            $mysqlst{'time'} = time();  
        }
        
    }
    for($request = $requests; $request; $request = $request->next()) {
      my $oid = $request->getOID();

      if ($request_info->getMode() == MODE_GET) {
        # if the requested oid is equals to ours, then return the data
        if ($oid == new NetSNMP::OID($mysql_at . ".2.1")) {
          $request->setValue(ASN_COUNTER64, $mysqlst{'select_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.2")) {
            $request->setValue(ASN_COUNTER64, $mysqlst{'update_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.3")) {
            $request->setValue(ASN_COUNTER64, $mysqlst{'insert_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.4")) {
            $request->setValue(ASN_COUNTER64, $mysqlst{'delete_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.5")) {
            $request->setValue(ASN_COUNTER64, $mysqlst{'replace_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.6")) {
            $request->setValue(ASN_COUNTER64, $mysqlst{'slowsql_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.7")) {
            $request->setValue(ASN_COUNTER64, $mysqlst{'fullscan_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.8")) {
            $request->setValue(ASN_INTEGER, $mysqlst{'slave_io'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.9")) {
            $request->setValue(ASN_INTEGER, $mysqlst{'slave_sql'});
        } 
      } elsif ($request_info->getMode() == MODE_GETNEXT) {
        # if the requested oid is lower than ours, then return ours
        if ($oid < new NetSNMP::OID($mysql_at . ".2.1")) {
          $request->setOID($mysql_at . ".2.1");
          $request->setValue(ASN_COUNTER64, $mysqlst{'select_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.1")) {
          $request->setOID($mysql_at . ".2.2");
          $request->setValue(ASN_COUNTER64, $mysqlst{'update_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.2")) {
            $request->setOID($mysql_at . ".2.3");
            $request->setValue(ASN_COUNTER64, $mysqlst{'insert_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.3")) {
            $request->setOID($mysql_at . ".2.4");
            $request->setValue(ASN_COUNTER64, $mysqlst{'delete_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.4")) {
            $request->setOID($mysql_at . ".2.5");
            $request->setValue(ASN_COUNTER64, $mysqlst{'replace_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.5")) {
            $request->setOID($mysql_at . ".2.6");
            $request->setValue(ASN_COUNTER64, $mysqlst{'slowsql_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.6")) {
            $request->setOID($mysql_at . ".2.7");
            $request->setValue(ASN_COUNTER64, $mysqlst{'fullscan_count'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.7")) {
            $request->setOID($mysql_at . ".2.8");
            $request->setValue(ASN_INTEGER, $mysqlst{'slave_io'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.8")) {
            $request->setOID($mysql_at . ".2.9");
            $request->setValue(ASN_INTEGER, $mysqlst{'slave_sql'});
        } elsif ($oid == new NetSNMP::OID($mysql_at . ".2.9")) {
            $request->setValue(ASN_NULL, "");
        } 
      }
    }

    print STDERR "  finished processing\n"
        if ($debugging);
}

sub other_handler {
    my ($handler, $registration_info, $request_info, $requests) = @_;
    my $request;

    print STDERR "refs: ",join(", ", ref($handler), ref($registration_info), 
                               ref($request_info), ref($requests)),"\n" if ($debugging);

    print STDERR "processing a request of type " . $request_info->getMode() . "\n"
        if ($debugging);
    my ($sec, $min, $hour, $mday, $mon, $year, $wday) = gmtime();
    my $hdcstat = 0;
    my $err = 0;
    if((stat("/data/logs/hdcenter.stat"))[7] > 0) {
        $hdcstat = 1;
    }
    for($request = $requests; $request; $request = $request->next()) {
      my $oid = $request->getOID();

      if ($request_info->getMode() == MODE_GET) {
        # if the requested oid is equals to ours, then return the data
        if ($oid == new NetSNMP::OID($other_at . ".2.1")) {
          $request->setValue(ASN_COUNTER64, $hdcstat);
        } 
      } elsif ($request_info->getMode() == MODE_GETNEXT) {
        # if the requested oid is lower than ours, then return ours
        if ($oid < new NetSNMP::OID($other_at . ".2.1")) {
          $request->setOID($other_at . ".2.1");
          $request->setValue(ASN_COUNTER64, $hdcstat);
        } elsif ($oid == new NetSNMP::OID($other_at . ".2.1")) {
            $request->setOID($other_at . ".2.2");  
          $request->setValue(ASN_COUNTER64, $hdcstat);
        } elsif ($oid == new NetSNMP::OID($other_at . ".2.2")) {
            $request->setValue(ASN_NULL, "");
        } 
      }
    }
}

sub get_history () {
    my $key = shift;
    my $rr = "";
    my $s = 0;
    my ($sec, $min, $hour, $mday, $mon, $year, $wday) = localtime();
    my $ldate = sprintf("%04d-%02d-%02d", $year+1900, $mon+1, $mday );
    my $record_file = $g_conf{'record_dir'} . $ldate . "_sys.info";
    if (-s $record_file) {
        $s = (stat($record_file))[7];
        open(DDD, $record_file) or $opf = 1;
    } else {
        my ($sec, $min, $hour, $mday, $mon, $year, $wday) = localtime(time()-3600*24);
        $ldate = sprintf("%04d-%02d-%02d", $year+1900, $mon+1, $mday);
        my $rfile = $g_conf{'record_dir'} . $ldate . "_sys.info";
        $s = (stat($rfile))[7];
        open(DDD, $rfile) or $opf = 1;
    }
    if ($opf == 0) {
        if ($s >= 2000) {
            seek(DDD,0,2);
            seek(DDD,-2000,1);
        }
        while(<DDD>) {
            if (/^$key=(.+)/o) {
                $rr = $1;
            }
        }
        close(DDD);
    }
    return $rr;
}

sub shut_it_down {
  $running = 0;
  print STDERR "shutting down\n" if ($debugging);
}
